;
; Virtual Disk Drive Project - March 22, 2002 - EJR (home)
;

;
; Layout of the i/o lines of a 74b used for the virtual drive
;

; RA0/AN0	- ADDR16 (ADDR14_17)	- Memory Addr 16 (out)
; RA1/AN1	- ADDR17 (ADDR14_17)	- Memory Addr 17 (out)
; RA2/AN2	- VD_WRITE_GATE		- VD Write Data Gate (in)
; RA3/AN3/VREF	- VD_HEAD_DIR		- VD Head Direction (in)
; RA4/T0CKI	- VD_HEAD_MOVE		- VD Head Move (in)
; RA5/SS/AN4	- VD_WRITE		- VD Write Data (in)
;
; RB0/INT	- ADDR8			- Memory address 8-15
; RB1		- ADDR9
; RB2		- ADDR10
; RB3		- ADDR11
; RB4(INT)	- ADDR12
; RB5(INT)	- ADDR13
; RB6(INT)	- ADDR14
; RB7(INT)	- ADDR15
; 
; RC0/T1OSO/T1CKI - VD_READ		- VD Read Data (out)
; RC1/T1OSI/CCP2 - VD_INDEX		- VD Index hole (out)
; RC2/CCP1	- VD_DRIVE_1		- VD Motor on with Drive 1 select (in)
; RC3/SCK/SCL	- VD_DRIVE_2		- VD Motor on with Drive 2 select (in)
; RC4/SDI/SDA	- VD_TRACK_0		- VD Track Zero detect (out)
; RC5/SDO	- WPROTECT		- write protect signal (out)
; RC6/TX/CK	- TX			- Serial transmit (out)
; RC7/RX/DT	- RX			- Serial Receive (in)
;
; RD0/PSP0	- DATA0 (DATA)		- Memory data 0 (in/out)
; RD1/PSP1	- DATA1 (DATA)		- Memory data 1 (in/out)
; RD2/PSP2	- DATA2 (DATA)		- Memory data 2 (in/out)
; RD3/PSP3	- DATA3 (DATA)		- Memory data 3 (in/out)
; RD4/PSP4	- DATA4 (DATA)		- Memory data 4 (in/out)
; RD5/PSP5	- DATA5 (DATA)		- Memory data 5 (in/out)
; RD6/PSP6	- DATA6 (DATA)		- Memory data 6 (in/out)
; RD7/PSP7	- DATA7 (DATA)		- Memory data 7 (in/out)
;
; RE0/RD/AN5	- MEM_LOW_INC		- Mem Addr 0-8 Inc (out)
; RE1/WR/AN6	- MEM_LOW_RESET		- Mem Addr 0-8 Reset (out)
; RE1/WR/AN6	- _MEM_OE		- Mem Out Enable (INVERTED) (out)
; RE2/CS/AN7	- _MEM_WRITE		- Mem Write (INVERTED) (out)

;;; NOTE - that the memory hi bits are in port A where of the 6 lines
;;;   only the bottom two (the memory lines) are outputs.  This makes it
;;;   easy to increment the memory address lines.

#define MEMBLOCKLO_PORT		PORTB
#define MEMBLOCKHI_PORT		PORTA
#define MEMBLOCKLO_CONF		TRISB
#define MEMBLOCKHI_CONF		TRISA

#define MEMBLOCKLO		MEMBLOCKLO_PORT
#define MEMBLOCKHI		MEMBLOCKHI_PORT

#define MEMBLOCKHI_16_CONF	MEMBLOCKHI_CONF,0x00
#define MEMBLOCKHI_17_CONF	MEMBLOCKHI_CONF,0x01

;;;
;;; MEM_CONTROL_PORT - controls external memory increment
;;;

#define MEM_CONTROL_PORT	PORTE
#define MEMINC			MEM_CONTROL_PORT,0x00
#define _MEMOE			MEM_CONTROL_PORT,0x01
#define _MEMWR			MEM_CONTROL_PORT,0x02

#define MEM_CONTROL_CONF	TRISE
#define MEMINC_CONF		MEM_CONTROL_CONF,0x00
#define _MEMOE_CONF		MEM_CONTROL_CONF,0x01
#define _MEMWR_CONF		MEM_CONTROL_CONF,0x02

;;;
;;; DATA_PORT
;;;
#define DATA_PORT	PORTD
#define DATA_CONF	TRISD

#define MEMDATA		DATA_PORT
#define MEMCTRL		DATA_CONF

;;;
;;; CONTROLS_PORT - includes incoming control signals and high addr bits
;;;
#define CONTROLS_PORT	PORTA
#define WRITE_GATE	PORTA,0x02
#define	HEAD_DIR	PORTA,0x03
#define HEAD_MOVE	PORTA,0x04
#define	WDATA		PORTA,0x05

#define CONTROLS_CONF	TRISA
#define WRITE_GATE_CONF	TRISA,0x02
#define	HEAD_DIR_CONF	TRISA,0x03
#define HEAD_MOVE_CONF	TRISA,0x04
#define	WDATA_CONF	TRISA,0x05

;;;
;;; CONTROLS2_PORT - more control signals
;;;
#define CONTROLS2_PORT		PORTC
#define RDATA			PORTC,0x00
#define INDEX_HOLE		PORTC,0x01
#define DSELECT_0		PORTC,0x02
#define DSELECT_1		PORTC,0x03
#define TRACK0			PORTC,0x04
#define	WPROTECT		PORTC,0x05

#define CONTROLS2_CONF		TRISC
#define RDATA_CONF		TRISC,0x00
#define INDEX_HOLE_CONF		TRISC,0x01
#define DSELECT_0_CONF		TRISC,0x02
#define DSELECT_1_CONF		TRISC,0x03
#define TRACK0_CONF		TRISC,0x04
#define	WPROTECT_CONF		TRISC,0x05

;;;*******************************************************************
;;; NAME:	MACRO RaiseNegLogic()   MACRO LowerNegLogic()
;;;		MACRO RaisePosLogic()   MACRO LowerPosLogic()
;;;
;;; DESCR:	Implements neg & pos logic for use in the Raise and Lower
;;;		macros.  These routines check to see if the given signal
;;;		was specified, and if so, implements the logic.  In this
;;;		way, the calling macro can just list the logic polarity
;;;		for each signal, and only one will be used at a time.
;;;
;;; ARGS:	
;;;
;;; RETURNS:	
;;;
;;; NOTES:	
;;;*******************************************************************

RaiseNegLogic:	macro	p1,p2,a,b
                if (p1 == a)  &&  (p2 == b)
		    bcf	p1,p2
                endif
		endm

LowerNegLogic:	macro	p1,p2,a,b
                if (p1 == a)  &&  (p2 == b)
		    bsf	p1,p2
                endif
		endm

LowerPosLogic:	macro	p1,p2,a,b
                if (p1 == a)  &&  (p2 == b)
		    bcf	p1,p2
                endif
		endm

RaisePosLogic:	macro	p1,p2,a,b
                if (p1 == a)  &&  (p2 == b)
		    bsf	p1,p2
                endif
		endm

;;;*******************************************************************
;;; NAME:	MACRO Raise()     MACRO Lower()
;;;
;;; DESCR:	Raise or Lower the given port line.  These macros
;;;		abstract the polarity of the logic.
;;;
;;; ARGS:	One of (only):
;;;		  RDATA
;;;		  INDEX_HOLE
;;;		  TRACK0
;;;		  WPROTECT
;;;
;;; RETURNS:	
;;;
;;; NOTES:	- These macros generates a SINGLE instruction so it is
;;;		  suitable for jumping over.
;;;		- each signal to be handled by this routine is listed
;;;		  along with its logic polarity.
;;;*******************************************************************
Raise:	macro arg1, arg2
        RaisePosLogic	RDATA,arg1,arg2
        RaisePosLogic	INDEX_HOLE,arg1,arg2
        RaisePosLogic	TRACK0,arg1,arg2
        RaisePosLogic	WPROTECT,arg1,arg2
	endm

Lower:  macro arg1, arg2
        LowerPosLogic	RDATA,arg1,arg2
        LowerPosLogic	INDEX_HOLE,arg1,arg2
        LowerPosLogic	TRACK0,arg1,arg2
        LowerPosLogic	WPROTECT,arg1,arg2
	endm

;;;*******************************************************************
;;; NAME:	MACRO SkpHighPosLogic()       MACRO SkpLowPosLogic()
;;;		MACRO SkpHighNegLogic()       MACRO SkpLowNegLogic()
;;;
;;; DESCR:	Implements neg & pos logic for checking incoming logic
;;;		lines.
;;;
;;; ARGS:	
;;;
;;; RETURNS:	
;;;
;;; NOTES:	
;;;*******************************************************************
SkpHighPosLogic:	macro	p1,p2,a,b
	                if (p1 == a)  &&  (p2 == b)
			    btfss	p1,p2
	                endif
			endm

SkpHighNegLogic:	macro	p1,p2,a,b
	                if (p1 == a)  &&  (p2 == b)
			    btfsc	p1,p2
	                endif
			endm
	
SkpLowPosLogic:		macro	p1,p2,a,b
	                if (p1 == a)  &&  (p2 == b)
			    btfsc	p1,p2
	                endif
			endm

SkpLowNegLogic:		macro	p1,p2,a,b
	                if (p1 == a)  &&  (p2 == b)
			    btfss	p1,p2
	                endif
			endm
	

;;;*******************************************************************
;;; NAME:	MACRO SkipIfHigh()    MACRO SkipIfLow()
;;;
;;; DESCR:	Used to implement checking the incoming lines with
;;;		adjustable logic polarity.
;;;
;;; ARGS:	Implements only:
;;;		DSELECT_0
;;;		DSELECT_1
;;;		WRITE_GATE
;;;		HEAD_DIR
;;;		HEAD_MOVE
;;;		WDATA
;;;
;;; RETURNS:	
;;;
;;; NOTES:	- implements a single instruction so that it can be
;;;		  jumped over
;;;*******************************************************************
SkipIfHigh:	macro	arg1, arg2
		SkpHighNegLogic	DSELECT_0,arg1,arg2
		SkpHighNegLogic	DSELECT_1,arg1,arg2
		SkpHighNegLogic	WRITE_GATE,arg1,arg2
		SkpHighNegLogic	HEAD_DIR,arg1,arg2
		SkpHighNegLogic	HEAD_MOVE,arg1,arg2
		SkpHighNegLogic	WDATA,arg1,arg2
		endm
		
SkipIfLow:	macro	arg1,arg2
		SkpLowNegLogic	DSELECT_0,arg1,arg2
		SkpLowNegLogic	DSELECT_1,arg1,arg2
		SkpLowNegLogic	WRITE_GATE,arg1,arg2
		SkpLowNegLogic	HEAD_DIR,arg1,arg2
		SkpLowNegLogic	HEAD_MOVE,arg1,arg2
		SkpLowNegLogic	WDATA,arg1,arg2
		endm

;;; =======================================================================
;;; =======================================================================

